function Assays = Read_assay_list(AssayListName)
% This function reads from the file AssayListName the list of assays to
% process, including the varied proteins in this assays, what the
% corresponding GTPase assay is, and where the data is localized, including
% in which tabs. This information is found in the tab 'Main'.
%
% Date: 16-11-2023

%% Main assay list
[~, ~, assay_table] = xlsread(AssayListName, 'Main');
% Find the row of the headers
row_header          = find(any(strcmp(assay_table, 'Assay name'), 2), 1);
% Find the columns where the different inputs are
col_assay           = find(strcmp(assay_table(row_header, :), 'Assay name'), 1);
col_proteins        = find(strcmp(assay_table(row_header, :), 'Proteins'), 1);
col_file            = find(strcmp(assay_table(row_header, :), 'Data file location'), 1);
col_tabs            = find(strcmp(assay_table(row_header, :), 'Relevant tab names'), 1);
col_ref_GTPase      = find(strcmp(assay_table(row_header, :), 'GTPase ref.'), 1);

% Find all the rows with inputs
rows_list           = row_header + find(cellfun(@(x) ~isempty(x) & ischar(x), assay_table(row_header + 1 : end, col_assay)));
% Find any columns with file names
file_columns        = col_file - 1 + find(any(cellfun(@(x) ~isempty(x) & ischar(x), assay_table(rows_list, col_file : col_tabs - 1))));
% Find any columns with tabs
tab_columns         = col_tabs - 1 + find(any(cellfun(@(x) ~isempty(x) & ischar(x), assay_table(rows_list, col_tabs : end))));

% Make the cell arrays with inputs
Assay_list          = assay_table(rows_list, col_assay);
Protein_list        = assay_table(rows_list, [col_proteins col_proteins + 1]);
Tab_list            = assay_table(rows_list, tab_columns);
File_list           = assay_table(rows_list, file_columns);
GTPase_ref_list     = assay_table(rows_list, col_ref_GTPase);

% Reduce the cell array with multiple columns to a row cell vector for
% proteins and tabs
[Protein_cell, file_cell, tab_cell] = deal(cell(size(Protein_list, 1), 1));
for i = 1 : numel(rows_list)
    Protein_cell{i} = Protein_list(i, cellfun(@(x) ~isempty(x) & ischar(x), Protein_list(i, :)))';
    file_cell{i}    = File_list(i,    cellfun(@(x) ~isempty(x) & ischar(x), File_list(i, :)))';
    tab_cell{i}     = Tab_list(i,     cellfun(@(x) ~isempty(x) & ischar(x), Tab_list(i, :)))';
end

% These lines temporarily append empty cells to the assay list. This is
% just to force Protein_cell, file_cell and tab_cell to get incorporated in
% the table Assays as cell arrays, not as strings.
Protein_cell{end + 1, 1}    = cell(2, 1);
file_cell{end + 1, 1}       = cell(2, 1);
tab_cell{end + 1, 1}        = cell(2, 1);
Assay_list{end + 1, 1}      = cell(1);
GTPase_ref_list{end + 1, 1} = cell(1);

% Convert all inputs to a single table
Assays              = cell2table([Protein_cell Assay_list file_cell tab_cell GTPase_ref_list], ...
                        'VariableNames', {'Proteins'; 'Label'; 'File'; 'Tabs'; 'GTPase_ref'});
% The last row we appended served its purpose and is now no longer needed
Assays              = Assays(1 : end - 1, :);
% Ensure all label are permitted, unique names for fields in structures
Assays.Label        = matlab.lang.makeValidName(matlab.lang.makeUniqueStrings(Assays.Label));